# pylint: disable=unused-import,abstract-method

##########################################################################################
#### Imports
##########################################################################################

from flask import Markup

import psynet.experiment
from psynet.field import claim_field
from psynet.participant import Participant, get_participant
from psynet.timeline import (
    Page,
    Timeline,
    PageMaker,
    CodeBlock,
    while_loop,
    conditional,
    switch
)
from psynet.page import (
    InfoPage,
    SuccessfulEndPage,
    NAFCPage,
    ModularPage,
    VolumeCalibration
)
from psynet.trial.non_adaptive import (
    NonAdaptiveTrialMaker,
    NonAdaptiveTrial,
    StimulusSet,
    StimulusSpec,
    StimulusVersionSpec
)
from controls import ChordRatingControl, ChordNAFCControl

from psynet.headphone import headphone_check

from timeline_parts import (
    instructions_experiment,
    final_questionnaire
)

import logging
logging.basicConfig(level=logging.INFO)
logger = logging.getLogger(__file__)

from stimulus_util import stimulus_set
from os.path import splitext
import random

import json

# import rpdb


#### GLOBAL ####
INITIAL_RECRUITMENT_SIZE = 20

##########################################################################################
#### Stimuli
##########################################################################################

class ChordTrial(NonAdaptiveTrial):
    __mapper_args__ = {"polymorphic_identity": "custom_trial"}

    def show_trial(self, experiment, participant):
        
        version = self.definition["version"]
        tag = self.definition["tag"]
        block = self.block

        specs = splitext(tag)[0] # remove extension
        [phase,source,frame] = specs.split("_")
        
        if not (phase=="II" and source=="mode"):
            prompt = Markup(
                f"""
                <style>
                    .text_prompt {{
                        text-align: center;
                    }}
                </style>

                <div class="text_prompt">
                    <p>
                        To what extent the sound you just heard matches the following word:
                    </p>
                    <p>
                        <strong>{block}</strong>
                    </p>
                </div>
                """
            )
            chords = [version[0]["intervals"]]

            page = ModularPage(
                "rating_trial",
                prompt,
                ChordRatingControl(chords,self.definition["f0"]),
                time_estimate=6
            )

        else:
            prompt = Markup(
                f"""
                <style>
                    .text_prompt {{
                        text-align: center;
                    }}
                </style>

                <div class="text_prompt">
                    <p>
                        Choose which sound best matches the following word:
                    </p>
                    <p>
                        <strong>{block}</strong>
                    </p>
                </div>
                """
            )

            chords = [vers["intervals"] for vers in version]

            page = ModularPage(
                "nafc_trial",
                prompt,
                ChordNAFCControl(chords,self.definition["f0"]),
                time_estimate=6
            )

        return page

    def make_definition(self, experiment, participant):
        definition = super().make_definition(experiment, participant)
        definition["f0"] = 60.0 + 10 * (random.random() - 0.5)
        return definition


class ChordTrialMaker(NonAdaptiveTrialMaker):
    consistency_check_type = "percent_agreement"
    give_end_feedback_passed = False
    performance_threshold = -1.0
    def compute_bonus(self, score, passed):
        if score is None:
            return 0.0
        else:
            return min(max(0.0, score),1.0) # chance-rate is 0.5



##########################################################################################
#### Experiment
##########################################################################################

# Weird bug: if you instead import Experiment from psynet.experiment,
# Dallinger won't allow you to override the bonus method
# (or at least you can override it but it won't work).
class Exp(psynet.experiment.Experiment):
    timeline = Timeline(
        VolumeCalibration(),
        headphone_check(),
        InfoPage("You passed the headphone screening task! Congratulations.", time_estimate=3),
        instructions_experiment,
        ChordTrialMaker(
            trial_class=ChordTrial,
            phase="experiment",
            stimulus_set=stimulus_set,
            time_estimate_per_trial=6,
            new_participant_group=True,
            max_trials_per_block=80,
            allow_repeated_stimuli=True,
            max_unique_stimuli_per_block=None,
            active_balancing_within_participants=True,
            active_balancing_across_participants=True,
            check_performance_at_end=True,
            check_performance_every_trial=False,
            target_num_participants=None,
            target_num_trials_per_stimulus=800,
            recruit_mode="num_trials",
            num_repeat_trials=3
        ),
        final_questionnaire,
        SuccessfulEndPage()
    )

    def __init__(self, session=None):
        super().__init__(session)
        self.initial_recruitment_size = INITIAL_RECRUITMENT_SIZE

extra_routes = Exp().extra_routes()
